/*
 * Copyright 2006-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.humuson.tms.batch.scheduler;

import java.util.List;
import java.util.Map;

import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;

import com.humuson.tms.batch.domain.TargetSchedule;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.TargetStatus;

/**
 * TargetSchedule job schedule
 * @author hyogun 
 * 
 */
public class TargetingScheduler extends AbstractScheduler<TargetSchedule> {

	private static Logger logger = LoggerFactory.getLogger(TargetingScheduler.class);

	@Override
	protected void scheduleExecute(JobExecutionContext context) {
		
		Map<String, Object> jobDataMap = context.getMergedJobDataMap();
		
		List<TargetSchedule> scheduleList = this.scheduleService.getScheduleList();
		if (logger.isDebugEnabled()) {
			logger.debug("Target ScheduleCount:{}", scheduleList.size());
		}
		
		long runTime = System.currentTimeMillis();
		
		for (TargetSchedule schedule : scheduleList) {
			
			if (scheduleService.isRunning(schedule.getMsgId())
					|| !jobDataMap.containsKey(schedule.getTargetType().toUpperCase())) {
				continue;
			}
			
			String jobName = jobDataMap.get(schedule.getTargetType().toUpperCase()).toString();
			
			jobDataMap.put(JobParamConstrants.REQ_TIME, runTime);
			jobDataMap.put(JobParamConstrants.SEND_LIST_TABLE, 
					StringUtils.getSendListTable(schedule.getPostId()));
			jobDataMap.put(JobParamConstrants.CHN_TYPE, schedule.getChannelType().getCode());
			jobDataMap.put(JobParamConstrants.POST_ID, schedule.getPostId());
			jobDataMap.put(JobParamConstrants.MSG_ID, schedule.getMsgId());
			jobDataMap.put(JobParamConstrants.TARGET_TYPE, schedule.getTargetType());
			jobDataMap.put(JobParamConstrants.DEDUPLICATION_YN, schedule.getDeduplicationYn());
			
			JobParameters jobParameters = getJobParametersFromJobMap(jobDataMap);
			
			boolean isExist = jobRepository.isJobInstanceExists(jobName, jobParameters);
			
			if (!isExist) {
				
				try {
					jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
					this.scheduleService.setRunning(schedule.getMsgId(), true);
					
					this.scheduleService.updateScheduleStatus(schedule.getMsgId(), TargetStatus.RUNNING.getCode());
				} catch (Exception e) {
					logger.error("Could not execute job.", e);
					this.scheduleService.updateScheduleStatus(schedule.getMsgId(), TargetStatus.ERROR.getCode());
					
					this.scheduleService.setRunning(schedule.getMsgId(), false);
				}
			}
		}	// end for loop
	}
}
