/*
 * Copyright 2006-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.humuson.tms.batch.scheduler;

import java.util.List;
import java.util.Map;

import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;

import com.humuson.tms.batch.domain.PushUpdateSchedule;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;

/**
 * update push result job schedule
 * @author hyogun 
 * 
 */
public class PushUpdateScheduler extends AbstractScheduler<PushUpdateSchedule> {

	private static Logger logger = LoggerFactory.getLogger(PushUpdateScheduler.class);
	
	@Override
	protected void scheduleExecute(JobExecutionContext context) {
		
		Map<String, Object> jobDataMap = context.getMergedJobDataMap();
		
		String jobName = (String) jobDataMap.get(JOB_NAME);
		
		List<PushUpdateSchedule> scheduleList = this.scheduleService.getScheduleList();
		if (logger.isDebugEnabled()) {
			logger.debug("Push Update ScheduleCount:{}", scheduleList.size());
		}
		
		long runTime = System.currentTimeMillis();
		for (PushUpdateSchedule schedule : scheduleList) {
			
			if (schedule.getMinId() == schedule.getMaxId()
					&& schedule.getMinId() == 0) {
				continue;
			}
			
			jobDataMap.put(JobParamConstrants.MAX_ID, schedule.getMaxId());
			jobDataMap.put(JobParamConstrants.MIN_ID, schedule.getMinId());
			jobDataMap.put(JobParamConstrants.REQ_TIME, runTime / 5000);
			
			JobParameters jobParameters = getJobParametersFromJobMap(jobDataMap);
			
			boolean isExist = jobRepository.isJobInstanceExists(jobName, jobParameters);
			
			if (!isExist && !scheduleService.isRunning(jobName)) {
				
				try {
					scheduleService.setRunning(jobName, true);
					jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
					
				} catch (Exception e) {
					logger.error("Could not execute job.", e);
					scheduleService.setRunning(jobName, false);
				}
			}
		}
	}
}
