/*
 * Copyright 2006-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.humuson.tms.batch.scheduler;

import java.util.List;
import java.util.Map;

import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;

import com.humuson.tms.batch.domain.PushMqReSendSchedule;
import com.humuson.tms.batch.domain.PushReSendSchedule;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;

/**
 * TargetSchedule job schedule
 * @author hyogun 
 * 
 */
public class PushMqReSendScheduler extends AbstractScheduler<PushMqReSendSchedule> {

	private static Logger logger = LoggerFactory.getLogger(PushMqReSendScheduler.class);
	
	@Override
	protected void scheduleExecute(JobExecutionContext context) {
		
		Map<String, Object> jobDataMap = context.getMergedJobDataMap();
		
		String jobName = (String) jobDataMap.get(JOB_NAME);
		
		List<PushMqReSendSchedule> scheduleList = this.scheduleService.getScheduleList();
		if (logger.isDebugEnabled()) {
			logger.debug("Push ReSend ScheduleCount:{}", scheduleList.size());
		}
		
		jobDataMap.put(JobParamConstrants.REQ_TIME, System.currentTimeMillis() / 10000);
		jobDataMap.put(JobParamConstrants.JOB_NAME, jobName);
		
		for (PushMqReSendSchedule schedule : scheduleList) {
				
			jobDataMap.put(JobParamConstrants.APP_GRP_KEY, schedule.getAppGrpKey());
			jobDataMap.put(JobParamConstrants.SERVER_ID, schedule.getServerId());
			jobDataMap.put(JobParamConstrants.SEND_TYPE, schedule.getSendType());
			jobDataMap.put(JobParamConstrants.APP_GRP_ID, schedule.getAppGrpId());
			
			JobParameters jobParameters = getJobParametersFromJobMap(jobDataMap);
			
			logger.info("push MQ resend jobParameters [{}]", jobParameters.toString());
			
			boolean isExist = jobRepository.isJobInstanceExists(jobName, jobParameters);
			
			if (!isExist && !scheduleService.isRunning(schedule.getKey())) {
				
				try {
					scheduleService.setRunning(schedule.getKey(), true);
					jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
					
				} catch (Exception e) {
					logger.error("Could not execute job.", e);
					scheduleService.setRunning(schedule.getKey(), false);
				}
					
			}
			
		}
	}
}
