package com.humuson.tms.batch.scheduler;

import java.util.List;
import java.util.Map;

import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;

import com.humuson.rainboots.datastore.DataStore;
import com.humuson.tms.batch.domain.PushFeedbackSchedule;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;

/**
 * @author 
 * 
 */
public class PushFeedbackScheduler extends AbstractScheduler<PushFeedbackSchedule> {

	private static Logger logger = LoggerFactory.getLogger(PushFeedbackScheduler.class);
	
	private StringRedisTemplate redisTemplate;
	
	
	private long getFeedBackCountByAppKey(String appKey){
		long feedbackSize = 0;
		try{
			StringBuilder sb = new StringBuilder();
			String key = sb.append(DataStore.KEY_FEEDBACK).append(DataStore.DELIM)
					.append(appKey).toString();
			
			feedbackSize = redisTemplate.opsForList().size(key);
		}catch(Exception e){ 
			logger.error("redis Feedback Scheduler error : {}", e);
		}
		
		return feedbackSize;
		
	}

	@Override
	protected void scheduleExecute(JobExecutionContext context) {
		
		Map<String, Object> jobDataMap = context.getMergedJobDataMap();
		
		String jobName = (String) jobDataMap.get(JOB_NAME);
			
		List<PushFeedbackSchedule> feedbackScheduleList = scheduleService.getScheduleList();
		
		redisTemplate = (StringRedisTemplate) ctx.getBean("redisTemplate");
		
		for (PushFeedbackSchedule feedbackSchedule : feedbackScheduleList) {
			long cnt = getFeedBackCountByAppKey(feedbackSchedule.getAppKey()) ;
			if (scheduleService.isRunning(feedbackSchedule.getAppKey())
					|| cnt <= 0) {
				logger.debug("TB_FEEDBACK_RAW [appKey:{}, size:{}] skip schedule!", 
						feedbackSchedule.getAppKey(), cnt);
				continue;
			}
			jobDataMap.put(JobParamConstrants.REQ_TIME, System.currentTimeMillis() / 10000);
			jobDataMap.put(JobParamConstrants.APP_KEY, feedbackSchedule.getAppKey());
			jobDataMap.put(JobParamConstrants.APP_GRP_ID, feedbackSchedule.getAppGrpId());
			JobParameters jobParameters = getJobParametersFromJobMap(jobDataMap);
			
			boolean isExist = jobRepository.isJobInstanceExists(jobName, jobParameters);
			
			if (!isExist) {
				
				try {
					jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
					scheduleService.setRunning(feedbackSchedule.getAppKey(), true);
					logger.debug("feedback job is launch!!!");
					
				} catch (Exception e) {
					logger.error("Could not execute job.", e);
					scheduleService.setRunning(feedbackSchedule.getAppKey(), false);
				}
			}
		}
	}
}
