/*
 * Copyright 2006-2007 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.humuson.tms.batch.scheduler;

import java.util.List;
import java.util.Map;

import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;

import com.humuson.tms.batch.domain.OptmzSchedule;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.StatusType;

/**
 * 수동 채널에 대한 optimz job schedule
 * @author hyogun 
 * 
 */
public class OptmzManualChnScheduler extends AbstractScheduler<OptmzSchedule> {

	private static Logger logger = LoggerFactory.getLogger(OptmzManualChnScheduler.class);

	@Override
	protected void scheduleExecute(JobExecutionContext context) {
		
		Map<String, Object> jobDataMap = context.getMergedJobDataMap();
		
		String jobName = (String) jobDataMap.get(JOB_NAME);
		
		List<OptmzSchedule> scheduleList = this.scheduleService.getScheduleList();
		
		if (logger.isDebugEnabled()) {
			logger.debug("Manual Channel ScheduleCount:{}", scheduleList.size());
		}
		
		long runTime = System.currentTimeMillis();
		
		for (OptmzSchedule schedule : scheduleList) {
			
			if (schedule.getPPostId().equals(schedule.getPostId())
					|| scheduleService.isRunning(schedule.getPostId())) {
				if (schedule.getPPostId().equals(schedule.getPostId())) {
					this.scheduleService.updateScheduleStatus(schedule.getPostId(), StatusType.COMPLETE.getCode());
				}
				continue;
			}
			
			jobDataMap.put(JobParamConstrants.REQ_TIME, runTime);
			jobDataMap.put(JobParamConstrants.CHN_TYPE, schedule.getChannelType());
			jobDataMap.put(JobParamConstrants.P_POST_ID, schedule.getPPostId());
			jobDataMap.put(JobParamConstrants.POST_ID, schedule.getPostId());
			jobDataMap.put(JobParamConstrants.MSG_ID, schedule.getMsgId());
			jobDataMap.put(JobParamConstrants.OPTMZ_TARGET_TYEPE, schedule.getOptmzTargetType());
			jobDataMap.put(JobParamConstrants.SEND_LIST_TABLE, StringUtils.getSendListTable(schedule.getPostId()));
			jobDataMap.put(JobParamConstrants.TARGET_LIST_TABLE, StringUtils.getSendListTable(schedule.getPPostId()));
			
			JobParameters jobParameters = getJobParametersFromJobMap(jobDataMap);
			
			boolean isExist = jobRepository.isJobInstanceExists(jobName, jobParameters);
			
			if (!isExist) {
				
				try {
					jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
					scheduleService.setRunning(schedule.getPostId(), true);
					this.scheduleService.updateScheduleStatus(schedule.getPostId(), StatusType.SENDING.getCode());
					
				} catch (Exception e) {
					logger.error("Could not execute job.", e);
					this.scheduleService.updateScheduleStatus(schedule.getPostId(), StatusType.ERROR.getCode());
				}
			}
		}
	}
}
