package com.humuson.tms.batch.scheduler;

import java.util.List;

import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;

import com.humuson.tms.batch.domain.FatigueFilterSchedule;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.batch.service.impl.FatigueFilterScheduleStatusUpdateService;
import com.humuson.tms.common.util.FatigueListTableNameUtil;
import com.humuson.tms.constrants.StatusType;

import lombok.extern.slf4j.Slf4j;

/**
 * 
 * @author hyogun
 * @see 피로도 필터링 스케쥴을 가지고 와서 job을 실행시켜줌
 *
 */
@Slf4j
public class FatigueFilterScheduler extends AbstractScheduler<FatigueFilterSchedule> {
	
	private static Logger logger = LoggerFactory.getLogger(FatigueFilterScheduler.class);
	
	FatigueFilterScheduleStatusUpdateService fatigueFilterScheduleStatusUpdateService;
	
	@Override
	public void init(JobExecutionContext context) throws JobExecutionException {
		super.init(context);
		fatigueFilterScheduleStatusUpdateService = ctx.getBean("fatigueFilterScheduleStatusUpdateService", FatigueFilterScheduleStatusUpdateService.class);
	}
	
	@Override
	protected void scheduleExecute(JobExecutionContext context) {
		
		JobDataMap jobDataMap = context.getMergedJobDataMap();
		
		Object[] params = new Object[]{StatusType.FATIGUE_FILTER_RESERVE.getCode(), StatusType.FATIGUE_FILTER_RESERVE.getCode()};
		
		List<FatigueFilterSchedule> scheduleList = this.scheduleService.getScheduleList(params);
		
		String jobName = jobDataMap.get(JOB_NAME).toString();
		
		long runTime = System.currentTimeMillis();
		
		for (FatigueFilterSchedule schedule : scheduleList) {
			
			jobDataMap.put(JobParamConstrants.REQ_TIME, runTime);
			jobDataMap.put(JobParamConstrants.SEND_TYPE, schedule.getSendType());
			jobDataMap.put(JobParamConstrants.POST_ID, schedule.getPostId());
			jobDataMap.put(JobParamConstrants.SERVER_ID, schedule.getServerId());
			jobDataMap.put(JobParamConstrants.SITE_ID, schedule.getSiteId());
			jobDataMap.put(JobParamConstrants.WORKDAY, schedule.getWorkday());
			jobDataMap.put(JobParamConstrants.CHN_TYPE, schedule.getChannelType().getCode());
			jobDataMap.put(JobParamConstrants.SEND_LIST_TABLE, schedule.getListTable());
			jobDataMap.put(JobParamConstrants.FATIGUE_LIST_TABLE, FatigueListTableNameUtil.getFatigueListTableName(schedule.getChannelType().getCode()));
			
			FatigueFilterSchedule status = new FatigueFilterSchedule();
			status.setSendType(schedule.getSendType());
			status.setPostId(schedule.getPostId());
			status.setServerId(schedule.getServerId());
			
			JobParameters jobParameters = getJobParametersFromJobMap(jobDataMap);
			
			boolean isExist = jobRepository.isJobInstanceExists(jobName, jobParameters);
			
			if (!isExist && !scheduleService.isRunning(schedule.getPostId())) {
				
				try {
					
					this.scheduleService.setRunning(schedule.getPostId(), true);
					status.setJobStatus(StatusType.FATIGUE_FILTER_ON.getCode());
					
					fatigueFilterScheduleStatusUpdateService.updateSchdlStatus(status);
					
					log.info("fatigue filter start. postId : {}, serverId : {}",schedule.getPostId(), schedule.getServerId());
					
					jobLauncher.run(jobLocator.getJob(jobName), jobParameters);
					
				} catch (Exception e) {
					
					logger.error("Could not execute job.", e);
					
					status.setJobStatus(StatusType.ERROR.getCode());
					fatigueFilterScheduleStatusUpdateService.updateSchdlStatus(status);
					this.scheduleService.setRunning(schedule.getPostId(), false);
				}
			}
		}	// end for loop
	}
}
