package com.humuson.tms.batch.scheduler;

import java.util.Date;
import java.util.Map;
import java.util.Map.Entry;

import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.configuration.JobLocator;
import org.springframework.batch.core.launch.JobLauncher;
import org.springframework.batch.core.launch.JobOperator;
import org.springframework.batch.core.repository.JobRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.scheduling.quartz.QuartzJobBean;

import com.humuson.tms.batch.service.ScheduleService;
import com.humuson.tms.common.util.StringUtils;

/**
 * Quartz 스케줄 처리를 담당하는 추상 클래스
 * @author hyogun
 *
 * @param <T>
 */
public abstract class AbstractScheduler<T> extends QuartzJobBean {
	
	private static Logger logger = LoggerFactory.getLogger(AbstractScheduler.class);
	
	protected JobLocator jobLocator;

	protected JobLauncher jobLauncher;
	
	protected JobRepository jobRepository;
	
	protected JobOperator jobOperator;
	protected ThreadPoolTaskExecutor excutor = null;
	
	protected ScheduleService<T> scheduleService;
	protected String scheduleServiceName;
	
	protected ApplicationContext ctx;
	
	protected static final String JOB_LOCATOR_CONTEXT_KEY = "jobLocator";
	protected static final String JOB_LAUNCHER_CONTEXT_KEY = "jobLauncher";
	protected static final String JOB_REPOSITORY_CONTEXT_KEY = "jobRepository";
	protected static final String JOB_OPERATOR_CONTEXT_KEY = "jobOperator";
	protected static final String APPLICATION_CONTEXT_KEY = "applicationContext";
	protected static final String JOB_NAME = "jobName";
	
	/**
	 * 
	 * @param context
	 * @throws JobExecutionException
	 */
	@SuppressWarnings("unchecked")
	public void init(JobExecutionContext context) 
			throws JobExecutionException {
		
		try {
			jobLocator = (JobLocator) context.getScheduler().getContext().get(JOB_LOCATOR_CONTEXT_KEY);  
			jobLauncher = (JobLauncher) context.getScheduler().getContext().get(JOB_LAUNCHER_CONTEXT_KEY);
			jobRepository = (JobRepository) context.getScheduler().getContext().get(JOB_REPOSITORY_CONTEXT_KEY);
			ctx = (ApplicationContext) context.getScheduler().getContext().get(APPLICATION_CONTEXT_KEY);
			
			excutor = ctx.getBean("threadPoolTaskExecutor", ThreadPoolTaskExecutor.class);
			
			if (logger.isDebugEnabled()) 
				logger.debug("threadPoolTaskExecutor queue size :{}", excutor.getThreadPoolExecutor().getActiveCount());
			
			if (!StringUtils.isNull(scheduleServiceName))
				scheduleService = (ScheduleService<T>) ctx.getBean(scheduleServiceName);
		} catch (SchedulerException e) {
			logger.error("Scheduler init error", e);
		}
	}
	
	protected abstract void scheduleExecute(JobExecutionContext context) throws JobExecutionException;
	
	@Override
	protected void executeInternal(JobExecutionContext context)
			throws org.quartz.JobExecutionException {
		init(context);
		scheduleExecute(context);
		
	}
	/*
	 * Copy parameters that are of the correct type over to
	 * {@link JobParameters}, ignoring jobName.
	 * 
	 * @return a {@link JobParameters} instance
	 */
	protected JobParameters getJobParametersFromJobMap(Map<String, Object> jobDataMap) {

		JobParametersBuilder builder = new JobParametersBuilder();

		for (Entry<String, Object> entry : jobDataMap.entrySet()) {
			String key = entry.getKey();
			Object value = entry.getValue();
			if (value instanceof String && !key.equals(JOB_NAME)) {
				builder.addString(key, (String) value);
			}
			else if (value instanceof Float || value instanceof Double) {
				builder.addDouble(key, ((Number) value).doubleValue());
			}
			else if (value instanceof Integer || value instanceof Long) {
				builder.addLong(key, ((Number)value).longValue());
			}
			else if (value instanceof Date) {
				builder.addDate(key, (Date) value);
			}
			else {
				logger.debug("JobDataMap contains values which are not job parameters (ignoring). [key:{}, value:{}]", key, value);
			}
		}
		
		return builder.toJobParameters();
	}
	
	@Autowired
	public void setScheduleServiceName(String scheduleServiceName) {
		this.scheduleServiceName = scheduleServiceName;
	}
}
