package com.humuson.tms.batch.job.tasklet;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SqlParameterValue;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterUtils;

import com.humuson.tms.batch.domain.FatigueFilterSchedule;
import com.humuson.tms.batch.domain.SiteFatigueSettingInfo;
import com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.constrants.CommonType;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class FatigueConditionCheckTasklet implements Tasklet, StepExecutionListener {
	
	private static final String UNDER_BAR = "_";
	
	private static final String REPLACE_ORACLE_INDEX_NAME = "@@INDEXNAME@@";
	
	private StepExecution stepExecution;
	
	private String upsertFatigueInfoSql;
	@Setter
	private String selectSiteFatigueFilterCntSql;
	@Setter
	private String upsertCampFatigueInfoSql;
	@Setter
	private String upsertAutoFatigueInfoSql;
	
	private String fatigueSendListSql;
	@Setter
	private String fatigueCampSendListSql;
	@Setter
	private String fatigueAutoSendListSql;
	
	@Value("#{dbConfig['tms.jdbc.type']}")
	protected String dbType;
	
	@Autowired
	private NamedParameterJdbcTemplate namedParameterJdbcTemplate;
	
	@Override
	public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception {
		
		JobParameters jobParameters = chunkContext.getStepContext().getStepExecution().getJobParameters();
		log.debug("execute()... JobParameters: {}", jobParameters);
		
		FatigueFilterSchedule schedule = new FatigueFilterSchedule();
		schedule.setSiteId(Integer.parseInt(jobParameters.getString(JobParamConstrants.SITE_ID)));
		schedule.setWorkday(jobParameters.getString(JobParamConstrants.WORKDAY));
		schedule.setChnType(jobParameters.getString(JobParamConstrants.CHN_TYPE));
		schedule.setServerId(jobParameters.getString(JobParamConstrants.SERVER_ID));
		
		// 다음 스텝에서 사용할 쿼리 바인딩 변수를 담음
		List<SqlParameterValue> sqlParameterList = new LinkedList<SqlParameterValue>();

		// 이름(:파라미터명)기반 파라미터 맵핑처리를 위하 변수를 담음.
		MapSqlParameterSource mapSqlParameterSource = new MapSqlParameterSource();
		
		String postId = jobParameters.getString(JobParamConstrants.POST_ID);
		String sendType = jobParameters.getString(JobParamConstrants.SEND_TYPE);
		
		SiteFatigueSettingInfo filterCnt = getFilterCnt();
		
		if (CommonType.MASS.getCode().equals(sendType)) {
			
			schedule.setPostId(postId);
			
			upsertFatigueInfoSql = upsertCampFatigueInfoSql;
			
			fatigueSendListSql = fatigueCampSendListSql;
			
		} else {
			
			String[] workdayAndSeqno = postId.split(UNDER_BAR);
			if (workdayAndSeqno.length < 2) {
				throw new Exception("Wrong Format AUTO SEND POST_ID.");
			}
			
			String workday = workdayAndSeqno[0];
			int seqno = Integer.parseInt(workdayAndSeqno[1]);
			
			schedule.setWorkday(workday);
			schedule.setSeqno(seqno);
			
			upsertFatigueInfoSql = upsertAutoFatigueInfoSql;
			
			fatigueSendListSql = fatigueAutoSendListSql;
		}
		
		mapSqlParameterSource.addValue("dayLimit", filterCnt.getDayLimit(), Types.INTEGER);
		mapSqlParameterSource.addValue("monthLimit", filterCnt.getMonthLimit(), Types.INTEGER);
		mapSqlParameterSource.addValue("siteId", schedule.getSiteId(), Types.INTEGER);
		mapSqlParameterSource.addValue("postId", postId, Types.VARCHAR);
		mapSqlParameterSource.addValue("workday", schedule.getWorkday(), Types.VARCHAR);
		mapSqlParameterSource.addValue("seqno", schedule.getSeqno(), Types.INTEGER);
		mapSqlParameterSource.addValue("channelType", schedule.getChnType(), Types.VARCHAR);
		mapSqlParameterSource.addValue("serverId", schedule.getServerId(), Types.VARCHAR);
		
		// 오라클일시 테이블명에 맞추어 인덱스 힌트 적용.
		if("ORACLE".equalsIgnoreCase(dbType)){
			fatigueSendListSql = fatigueSendListSql.replace(REPLACE_ORACLE_INDEX_NAME, jobParameters.getString(JobParamConstrants.SEND_LIST_TABLE).replace("TMS", "PK"));
			upsertFatigueInfoSql = upsertFatigueInfoSql.replace(REPLACE_ORACLE_INDEX_NAME, jobParameters.getString(JobParamConstrants.SEND_LIST_TABLE).replace("TMS", "PK"));
		}
		
		// 피로도 발송건수를 올린다.
		//namedParameterJdbcTemplate.update(upsertFatigueInfoSql, new BeanPropertySqlParameterSource(schedule));
		
		// 이름(:파라미터명) 기반으로 파라미터 출현 순서에 맞춰 List에 적재. 
		// 다음 스텝에서 ListPreparedStatementSetter 에 전달하여 바인딩처리.
		List<Object> parameterArrayList = Arrays.asList(NamedParameterUtils.buildValueArray(fatigueSendListSql,mapSqlParameterSource.getValues()));
		
		// 이름(:파라미터명)을 PreparedStatments에 사용할 ? 로 변경.
		fatigueSendListSql = NamedParameterUtils.parseSqlStatementIntoString(fatigueSendListSql);
		
		stepExecution.getJobExecution().getExecutionContext().put(JobExecutionContextConstrants.KEY_FATIGUE_SEND_LIST_SQL, fatigueSendListSql);
//		stepExecution.getJobExecution().getExecutionContext().put(JobExecutionContextConstrants.KEY_FATIGUE_SEND_LIST_PARAMETERS, sqlParameterList);
		stepExecution.getJobExecution().getExecutionContext().put(JobExecutionContextConstrants.KEY_FATIGUE_SEND_LIST_PARAMETERS, parameterArrayList);
		
		
		log.debug("execute()... fatigueSendListSql: {}", fatigueSendListSql);
		log.debug("execute()... upsertFatigueInfoSql: {}", upsertFatigueInfoSql);
		
		return RepeatStatus.FINISHED;
	}
	
	@Override
	public void beforeStep(StepExecution _stepExecution) {
		this.stepExecution = _stepExecution;
		
	}
	
	public SiteFatigueSettingInfo getFilterCnt() {
		
		return namedParameterJdbcTemplate.query(selectSiteFatigueFilterCntSql, new RowMapper<SiteFatigueSettingInfo>() {
			@Override
			public SiteFatigueSettingInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
				SiteFatigueSettingInfo info = new SiteFatigueSettingInfo();
				info.setDayLimit(rs.getInt(SiteFatigueSettingInfo.KEY_FATIGUE_DAY_LIMIT));
				info.setMonthLimit(rs.getInt(SiteFatigueSettingInfo.KEY_FATIGUE_MONTH_LIMIT));
				return info;
			}
		}).get(0);
	}
	
	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		return stepExecution.getExitStatus();
	}
}
