package com.humuson.tms.batch.job.partition;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.batch.core.partition.support.Partitioner;
import org.springframework.batch.item.ExecutionContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;

public class MinMaxRangePartitioner implements Partitioner {

	private static Logger logger = LoggerFactory.getLogger(MinMaxRangePartitioner.class);
	private static final int PARTITOINS_PER_NODE = 5;
	private static final int MIN_COUNT_PER_NODE = 100;
	
	private Long minId;
	private Long maxId;
	private Long scheduleId;
	private String sql;
	
	@Autowired private JdbcTemplate jdbcTemplate;

	/**
	 * Partition a database table assuming that the data in the column specified
	 * are uniformly distributed. The execution context values will have keys
	 * <code>minValue</code> and <code>maxValue</code> specifying the range of
	 * values to consider in each partition.
	 *
	 * @see Partitioner#partition(int)
	 */
	@Override
	public Map<String, ExecutionContext> partition(int gridSize) {
		
		if (maxId == null || minId == null) {
			logger.info("maxId:{}, minId:{}", maxId, minId);
			if (sql != null && scheduleId != null) {
				Map<String, Object> map = jdbcTemplate.queryForMap(sql, scheduleId);
				if (minId == null)
					minId = Long.parseLong(map.get("MIN_ID").toString());
				if (maxId == null)
					maxId = Long.parseLong(map.get("MAX_ID").toString());
				logger.info("jdbc select [maxId:{}, minId:{}]", maxId, minId);
			}
		}
		
		long targetSize = maxId - minId;
		long targetSizePerNode = (targetSize / (gridSize * PARTITOINS_PER_NODE)) + 1;
		
		
		if (targetSizePerNode <= MIN_COUNT_PER_NODE) {
			targetSizePerNode = MIN_COUNT_PER_NODE;
		}
		
		logger.info("gridSize :{}, targetSize:{}, targetSizePerNode:{}", gridSize, targetSize, targetSizePerNode);
		
		Map<String, ExecutionContext> result = new HashMap<String, ExecutionContext>();
		int number = 0;
		long start = minId;
		long end = start + targetSizePerNode - 1;
		while (start <= maxId) {

			ExecutionContext value = new ExecutionContext();
			result.put("partition" + number, value);

			if (end >= maxId) {
				end = maxId;
			}
			value.putLong("partition.min.value", start);
			value.putLong("partition.max.value", end);
			start += targetSizePerNode;
			end += targetSizePerNode;
			number++;
		}
		logger.info("execution partition:{}", result.toString());
		return result;

	}

	public void setMaxId(Long maxId) {
		this.maxId = maxId;
	}

	public void setMinId(Long minId) {
		this.minId = minId;
	}

	public void setSql(String sql) {
		this.sql = sql;
	}
	public void setScheduleId(Long scheduleId) {
		this.scheduleId = scheduleId;
	}
	
	public static void main(String[] args) {
		MinMaxRangePartitioner partitioner = new MinMaxRangePartitioner();
		partitioner.setMinId(1L);
		partitioner.setMaxId(500L);
		partitioner.partition(5);
	}
}
