package com.humuson.tms.batch.job.partition;

import java.util.HashMap;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

import org.springframework.batch.core.partition.support.Partitioner;
import org.springframework.batch.item.ExecutionContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.StringUtils;

@Slf4j
public class JdbcColumnRangePartitioner implements Partitioner {

	private static final int PARTITOINS_PER_NODE = 5;
	private static final int MIN_COUNT_PER_NODE = 1000;

	@Autowired
	private JdbcTemplate jdbcTemplate;

	private String fromClause;

	private String column;

	private String whereClause;

	/**
	 * Partition a database table assuming that the data in the column specified
	 * are uniformly distributed. The execution context values will have keys
	 * <code>minValue</code> and <code>maxValue</code> specifying the range of
	 * values to consider in each partition.
	 *
	 * @see Partitioner#partition(int)
	 */
	@Override
	public Map<String, ExecutionContext> partition(int gridSize) {

		int partitionCount = gridSize * PARTITOINS_PER_NODE;
		String minQuery = "SELECT MIN(" + column + ") from " + fromClause;
		String maxQuery = "SELECT MAX(" + column + ") from " + fromClause;

		if(StringUtils.hasLength(whereClause)) {
			minQuery = minQuery + " WHERE " + whereClause;
			maxQuery = maxQuery + " WHERE " + whereClause;
		}
		log.info("minQuery : {}", minQuery);
		log.info("maxQuery : {}", maxQuery);
		int min = jdbcTemplate.queryForObject(minQuery, Integer.class);
		int max = jdbcTemplate.queryForObject(maxQuery, Integer.class);
		int targetSize = (max - min) / partitionCount * MIN_COUNT_PER_NODE + 1;

		Map<String, ExecutionContext> result = new HashMap<String, ExecutionContext>();
		int number = 0;
		int start = min;
		int end = start + targetSize - 1;
		log.info("TB_SEND_RAW [min:{}, max:{}]", min, max);
		while (start <= max) {

			ExecutionContext value = new ExecutionContext();
			result.put("partition" + number, value);

			if (end >= max) {
				end = max;
			}
			value.putLong("minValue", start);
			value.putLong("maxValue", end);
			start += targetSize;
			end += targetSize;
			number++;
		}

		return result;

	}

	public void setFromClause(String fromClause) {
		this.fromClause = fromClause;
	}

	public void setColumn(String column) {
		this.column = column;
	}

	public void setWhereClause(String whereClause) {
		this.whereClause = whereClause;
	}

}
