package com.humuson.tms.batch.item.redis;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import org.json.simple.JSONObject;
import org.json.simple.JSONValue;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemReader;
import org.springframework.batch.item.NonTransientResourceException;
import org.springframework.batch.item.ParseException;
import org.springframework.batch.item.UnexpectedInputException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.StringRedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;

import com.humuson.rainboots.datastore.DataStore;
import com.humuson.rainboots.proto.messages.FeedbackProtos.FeedbackResponse.MessageResultType;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.common.util.StringUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * 
 * @author hyogun
 *
 */
@Slf4j
public class PrivateFeedbackRedisItemReader implements ItemReader<PushResult>,  StepExecutionListener {
	
	private List<PushResult> results;
	
	private int currentReadCount = 0;
	private int currentItemCount = 0;
	private int maxItemCount = Integer.MAX_VALUE;
	
	@Value("#{dbConfig['commit.interval']}")
	private int feedbackCount;
	@Autowired
	private RedisTemplate<String, String> redisTemplate;
	
	private String appKey;
	private long appGrpId;
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		appKey = stepExecution.getJobParameters().getString(JobParamConstrants.APP_KEY);
		appGrpId = stepExecution.getJobParameters().getLong(JobParamConstrants.APP_GRP_ID);
	}
	
	@Override
	public PushResult read() throws Exception, UnexpectedInputException, ParseException,
			NonTransientResourceException {
		if (currentItemCount >= maxItemCount) {
			return null;
		}
		
		return doRead();
	}
	
	private synchronized PushResult doRead() throws Exception {
		if (results == null || currentReadCount >= currentItemCount) {
			
			doMoreFeedbackList();
			
			if (results != null && log.isDebugEnabled()) {
				log.debug("reader size : {}", currentItemCount);
			}
		}
		
		int next = currentReadCount++;
		
		if (results != null && next < results.size()) {
			return results.get(next);
		} else {
			return null;
		}
	}
	
	private List<String> getFeedbackList(final int limitCount) {
		List<String> returnList = new ArrayList<String>();
		long startTime = System.currentTimeMillis();
		try {
			StringBuilder sb = new StringBuilder();
			final String key = sb.append(DataStore.KEY_FEEDBACK).append(DataStore.DELIM)
					.append(appKey).toString();
			
			final long feedbackSize = redisTemplate.opsForList().size(key);
			
			if (feedbackSize > 0) {
				List<Object> results = redisTemplate.executePipelined(new RedisCallback<Object>(){

					@Override
					public Object doInRedis(RedisConnection connection)
							throws DataAccessException {
						StringRedisConnection stringRedisConnection = (StringRedisConnection) connection;
						long batchSize = limitCount;
						if (feedbackSize < limitCount) {
							batchSize = feedbackSize;
						}
						
						for (int i=0; i<batchSize; i++) {
							stringRedisConnection.lPop(key);
						}
						return null;
					}
				});
				
				for (Object obj : results) {
					returnList.add(obj.toString());
				}
			}

		} catch (Exception e) {
			log.error("error : {}", e);
		} finally {
			log.debug("REDIS --> popFeedbackList elapsed time : {}", System.currentTimeMillis() - startTime);
		}
		
		return returnList;
	}
	
	private void doMoreFeedbackList() {
		
		if (results == null) {
			results = new CopyOnWriteArrayList<PushResult>();
			currentItemCount = 0;
		} 
		
		final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
		
		List<String> feedbackList = getFeedbackList(feedbackCount);
		
		if (feedbackList != null && !feedbackList.isEmpty()) {
			
			log.info("appKey : {}, res size : {}",appKey, feedbackList.size());
		
			String reqeustId = null;
			for( String jsonStr : feedbackList) {
				try {
					JSONObject json = (JSONObject) JSONValue.parse(jsonStr);
					
					int status = MessageResultType.UNKNOWN_ERROR_VALUE;
					
					try {
						status = Integer.parseInt((String)json.get(DataStore.FEEDBACK_STATUS));
					} catch (Exception e) {
						log.error("Integer.parseInt error : [jsonStr:{}]", json.toString());
					}
					
					String recvTime = (String)json.get(DataStore.FEEDBACK_RECV_TIME);
					String sendTime = (String)json.get(DataStore.FEEDBACK_SEND_TIME);
					
					if (!StringUtils.isNull(recvTime)) {
						try {
							recvTime = sdf.format(new Date(Long.parseLong(recvTime)));
						} catch (Exception e) {
							log.error("recvTime parse error [recvTime:{}]", recvTime);
							recvTime = null;
						}
					}
						
					if (!StringUtils.isNull(sendTime)) {
						try {
							sendTime = sdf.format(new Date(Long.parseLong(sendTime)));
						} catch (Exception e) {
							log.error("sendTime error [recvTime:{}]", recvTime);
							sendTime = null;
						}
					}

					reqeustId = (String)json.get(DataStore.FEEDBACK_REQ_ID);
					
					if (reqeustId != null) {
						results.add(new PushResult((int)appGrpId, 
								String.valueOf(status), reqeustId, App.ANDROID,null)
								.setRecvTime(recvTime)
								.setSendTime(sendTime));
						currentItemCount++;
					} else {
						log.error("requestId is null [{}]", json.toString());
					}
				} catch (Exception e) {
					log.error("FEEDBACK parse error skip feedback info : {}", jsonStr);
				}
			}
		}
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		log.info("step read count : {}, currentReadCount : {}, currentItemCount : {} result size : {}", 
				stepExecution.getReadCount(), currentReadCount, currentItemCount, results == null ? 0 : results.size());
		return stepExecution.getExitStatus();
	}
}