package com.humuson.tms.batch.item.database;

import static com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants.KEY_ENCRYPT_YN;
import static com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants.KEY_ID_TARGET_YN;
import static com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants.KEY_MKT_YN;
import static com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants.KEY_SITE_ID;
import static com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants.KEY_SEQNO;
import static com.humuson.tms.batch.job.constrants.JobParamConstrants.CHN_TYPE;
import static com.humuson.tms.batch.job.constrants.JobParamConstrants.MSG_ID;
import static com.humuson.tms.batch.job.constrants.JobParamConstrants.TARGET_TYPE;


import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemProcessor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;

import com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.batch.service.SystemCodeService;
import com.humuson.tms.common.model.target.TargetData;
import com.humuson.tms.common.security.HumusonDecryptor;
import com.humuson.tms.common.security.HumusonEncryptor;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.ChannelType;
import com.humuson.tms.service.MemberFieldSecurityService;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 각 채널 타겟팅 추출 데이터에 대한 검증 클래스
 * @author hyogun
 *
 */
@Slf4j
public class TargetProcessor implements ItemProcessor<TargetData, TargetData>, StepExecutionListener {
	
	protected String firstChnType;
	protected String chnType;
	protected String msgId;
	protected String postId;
	protected String targetType;
	protected String idTargetYn;
	protected String abTestYn;
	protected Long targetCount;
	protected int abTestCount;
	protected String abTestRate;
	protected String sourceTargetEncryptYn;
	protected String dbEncryptFlag;
	protected List<String> channelList;
	protected String mktYn;
	private String addSiteUserYn;
	protected String deduplicationYn;
	protected String seqNo;
	
	@Setter private String updateFilterCount;
	@Setter private String selectAbTestInfo;
	@Setter private String selectChnList;
	@Setter private String selectAppDevice;
	@Setter private String selectUnLoginAppDevice;
	@Setter private String selectSiteUser;
	@Setter private String insertTargetErrorList;
	
	@Autowired protected SystemCodeService systemCodeServiceImpl;
	@Autowired protected MemberFieldSecurityService memberFieldSecurityServiceImpl; 
	@Autowired protected JdbcTemplate jdbcTemplate;
	@Value("#{config['tms.db.enc.key']}") protected String encKey;
	
	protected Map<String, Integer> filterMap;
	protected Map<String, Integer> readerMap;
	protected Map<String, Integer> writerMap;
	
	protected AtomicInteger readerCount = new AtomicInteger();
	protected AtomicInteger errorCount = new AtomicInteger();
	
	protected int maxChunkSize = 10;
	protected int siteId = -1;
	protected int appGrpId = -1;
	private Random rnd = new Random();
	
	/**
	 * 타겟 대상 item에 대한 검증 메소스 
	 *  return값이 false인 경우 filter 처리됨
	 * @param item
	 * @return boolean
	 * @throws Exception
	 */
	protected boolean checkValidation(TargetData item) throws Exception {
		
		boolean isValidation = true; 
		
		if (StringUtils.isNull(item.getMemberId())) {
			errorCount.incrementAndGet();
			log.info("TMS_M_ID is null [{}]", item.toString());
			return false;
		}
		
		if ( ChannelType.PUSH.getCode().equals(firstChnType) ) {
			
			if ( ( "Y".equals(mktYn) && StringUtils.isNull(item.getMktYn()) )
					|| StringUtils.isNull(item.getMemberToken())
					|| StringUtils.isNull(item.getDeviceId())) {
				
				try {
					Map<String, Object> map;
			
					if(item.getMemberId().startsWith("TMS-")){ 
						map	= jdbcTemplate.queryForMap(selectUnLoginAppDevice, siteId, item.getMemberId(),appGrpId);
					}else{
						map	= jdbcTemplate.queryForMap(selectAppDevice, siteId, item.getMemberId(),appGrpId);
					}
					
					
					if ( StringUtils.isNull(item.getMemberToken()) ) {
						item.setMemberToken(map.get(TargetData.TMS_M_TOKEN).toString());
					}
					
					if ( StringUtils.isNull(item.getDeviceId()) ) {
						item.setDeviceId(map.get(TargetData.DEVICE_ID).toString());
					}
					
					if ( StringUtils.isNull(item.getMktYn()) ) {
						item.setMktYn(map.get(TargetData.MKT_YN).toString());
					}
					
				} catch (Exception e) {
					log.error("no exists appDevice [memberId:{}], error:{}", item.getMemberId(), e.getMessage());
					return false;
				}
			}
		} else if (ChannelType.EMAIL.getCode().equals(firstChnType)) {
			
			if (  "Y".equals(mktYn)
					|| StringUtils.isNull(item.getMemberEmail()) ) {
				
				try {
					
					Map<String, Object> map	= jdbcTemplate.queryForMap(selectSiteUser, siteId, item.getMemberId());
					
					if ( StringUtils.isNull(item.getMktYn()) ) {
						item.setMktYn(map.get(TargetData.MKT_YN).toString());
					}
					
					if ( StringUtils.isNull(item.getMemberName()) ) {
						item.setMemberName(map.get(TargetData.TMS_M_NAME).toString());
					}
					
					if ( StringUtils.isNull(item.getMemberEmail()) ) {
						item.setMemberEmail(map.get(TargetData.TMS_M_EMAIL).toString());
					}
					
				} catch (Exception e) {
					if (!"Y".equals(addSiteUserYn)) {
						log.error("no exists siteUser [siteId:{}, memberId:{}], error:{}]", siteId, item.getMemberId(), e.getMessage());
						return false;
					}
				}
			}
			
		} else if ( ChannelType.SMS.getCode().equals(firstChnType) ) {
			
			if ( "Y".equals(mktYn)
					|| StringUtils.isNull(item.getMemberPhone()) ) {
				
				try {
					
					Map<String, Object> map	= jdbcTemplate.queryForMap(selectSiteUser, siteId, item.getMemberId());
					
					if ( StringUtils.isNull(item.getMktYn()) ) {
						item.setMktYn(map.get(TargetData.MKT_YN).toString());
					}
					
					if ( StringUtils.isNull(item.getMemberName()) ) {
						item.setMemberName(map.get(TargetData.TMS_M_NAME).toString());
					}
					
					if ( StringUtils.isNull(item.getMemberPhone()) ) {
						item.setMemberPhone(map.get(TargetData.TMS_M_PHONE).toString());
					}
					
				} catch (Exception e) {
					if (!"Y".equals(addSiteUserYn)) {
						log.error("no exists siteUser [siteId:{}, memberId:{}], error:{}]", siteId, item.getMemberId(), e.getMessage());
						return false;
					}
				}
				
			}
		}
		
		if ("Y".equals(mktYn) && "N".equals(item.getMktYn())) {
			log.error("mkt flag 'N' [tms_m_id:{}, tms_m_name:{}]", item.getMemberId(), item.getMemberName());
			return false;
		} else {
			
			if (channelList.contains(ChannelType.EMAIL.getCode())) {
				isValidation = this.checkValidationEmailChannel(item);
			}
			
			if (channelList.contains(ChannelType.SMS.getCode())) {
				isValidation = this.checkValidationSmsChannel(item);
			}
			
			if (channelList.contains(ChannelType.PUSH.getCode())) {
				isValidation = this.checkValidationPushChannel(item);
			}
			
			return isValidation;
		}
	}

	private boolean checkValidationSmsChannel(TargetData item) {
		String phone = item.getMemberPhone();
		if (phone == null) {
			return false;
		}
		
		if ("Y".equals(sourceTargetEncryptYn)) {
			try {
				phone = HumusonDecryptor.decrypt(item.getMemberPhone(), encKey, false);
			} catch (Exception e) {
				log.error("sms decrypt error [{}] msg:{}", phone, e.getMessage());
			}
		}
		
		phone = phone.replaceAll("-", "");
		
		if (ChannelType.SMS.getCode().equals(firstChnType)
				&& StringUtils.isSmsError(phone)) {
			return false;
		}
		
		item.setTelCode(StringUtils.getTeleCom(phone));
		
		if ("Y".equals(dbEncryptFlag)) {
			item.setMemberPhone(HumusonEncryptor.encrypt(phone, encKey, false));
		} else {
			item.setMemberPhone(phone);
		}
		
		return true;
	}

	private boolean checkValidationPushChannel(TargetData item) {
		if (ChannelType.PUSH.getCode().equals(firstChnType)
				&& (StringUtils.isNull(item.getDeviceId())
					|| StringUtils.isNull(item.getMemberToken()))) {
			try {
				Map<String, Object> map	= jdbcTemplate.queryForMap(selectAppDevice, siteId, item.getMemberId(),appGrpId);
				
				if ( StringUtils.isNull(item.getMemberToken()) ) {
					item.setMemberToken(map.get(TargetData.TMS_M_TOKEN).toString());
				}
				
				if ( StringUtils.isNull(item.getDeviceId()) ) {
					item.setDeviceId(map.get(TargetData.DEVICE_ID).toString());
				}
				
				if ( StringUtils.isNull(item.getMktYn()) ) {
					item.setMktYn(map.get(TargetData.MKT_YN).toString());
				}
				
			} catch (Exception e) {
				log.error("no exists appDevice [memberId:{}], error:{}", item.getMemberId(), e.getMessage());
				return false;
			}
		}
		return true;
	}

	private boolean checkValidationEmailChannel(TargetData item) {
		String email = item.getMemberEmail();
		if ("Y".equals(sourceTargetEncryptYn) || email.indexOf('@') < 0) {
			try {
				email = HumusonDecryptor.decrypt(item.getMemberEmail(), encKey, false);
			} catch (Exception e) {
				log.error("email decrypt error [{}] msg:{}", email, e.getMessage());
			}
		}
		
		if (ChannelType.EMAIL.getCode().equals(firstChnType)
				&& StringUtils.isEmailError(email)) {
			return false;
		} 
		
		item.setDomain(StringUtils.getDomain(email));
		
		if ("Y".equals(dbEncryptFlag)) {
			item.setMemberEmail(HumusonEncryptor.encrypt(email, encKey, false));
		} else {
			item.setMemberEmail(email);
		}
		
		return true;
	}
	
	@Override
	public TargetData process(TargetData item) throws Exception {
		try {
			if (!this.checkValidation(item)) {
				// 타게팅 실패리스트 INSERT (IE:유효성 에러 OE:기타에러 DE:중복에러)
				jdbcTemplate.update(insertTargetErrorList, firstChnType, msgId, seqNo, item.getMemberId(), firstChnType.equals("EM") ? item.getMemberEmail() : firstChnType.equals("SM") ? item.getMemberPhone(): item.getDeviceId(),"IE");
				this.increamentFilterCnt();
				return null;
			}
		} catch (Exception e) {
			log.error("CheckValidation error [item:{}], msg:{}", item.toString(), e.getMessage());
			return null;
		}
			
		readerCount.incrementAndGet();
		
		item.setChannelType(firstChnType);
		item.setSiteId(siteId);
		item.setChunkId(String.valueOf(rnd.nextInt(maxChunkSize)));
		item.setMsgId(msgId);
		item.setSeqNo(seqNo);
		
		if ("Y".equals(deduplicationYn)) {
			if (ChannelType.EMAIL.getCode().equals(firstChnType)) {
				item.setDuplicateKey(item.getMemberEmail());
			} else if (ChannelType.SMS.getCode().equals(firstChnType)) {
				item.setDuplicateKey(item.getMemberPhone());
			} else if (ChannelType.PUSH.getCode().equals(firstChnType)) {
				item.setDuplicateKey(item.getDeviceId());
			}
		}
		
		return item;
	}
	
	protected synchronized void increamentFilterCnt() {
		Integer filterCnt = filterMap.get(chnType);
		if (filterCnt == null) {
			filterCnt = 0;
		}
		
		filterMap.put(chnType, ++filterCnt);
	}
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		filterMap = new ConcurrentHashMap<String, Integer>();
		readerMap = new ConcurrentHashMap<String, Integer>();
		writerMap = new ConcurrentHashMap<String, Integer>();
		
		chnType = stepExecution.getJobParameters().getString(CHN_TYPE);
		msgId = stepExecution.getJobParameters().getString(MSG_ID);
		postId = stepExecution.getJobParameters().getString(JobParamConstrants.POST_ID);
		targetType = stepExecution.getJobParameters().getString(TARGET_TYPE);
		idTargetYn = stepExecution.getJobExecution().getExecutionContext().getString(KEY_ID_TARGET_YN, "N");
		sourceTargetEncryptYn = stepExecution.getJobExecution().getExecutionContext().getString(KEY_ENCRYPT_YN, "N");
		siteId = stepExecution.getJobExecution().getExecutionContext().getInt(KEY_SITE_ID, 0);
		firstChnType = stepExecution.getJobExecution().getExecutionContext().getString(JobExecutionContextConstrants.KEY_FIRST_CHANNEL_TYPE);
		mktYn = stepExecution.getJobExecution().getExecutionContext().getString(KEY_MKT_YN, "N");
		dbEncryptFlag = systemCodeServiceImpl.getDbEncryptYn();
		addSiteUserYn = stepExecution.getJobExecution().getExecutionContext()
				.getString(JobExecutionContextConstrants.KEY_ADD_SITE_USER_YN, "N");
		deduplicationYn = stepExecution.getJobParameters().getString(JobParamConstrants.DEDUPLICATION_YN);
		seqNo = stepExecution.getJobExecution().getExecutionContext().getString(KEY_SEQNO);
		
		channelList = jdbcTemplate.queryForList(selectChnList, new Object[]{msgId}, String.class);
		
		appGrpId = stepExecution.getJobExecution().getExecutionContext().getInt(JobExecutionContextConstrants.KEY_APP_GRP_ID);
		
		log.info("siteId :{}, msgId:{} targetType:{}, idTargetYn:{}, dbEncryptFlag:{}, sourceEncryptYn:{}, deduplicationYn:{}, appGrpId:{}", 
				siteId, msgId, targetType, idTargetYn, dbEncryptFlag, sourceTargetEncryptYn, deduplicationYn, appGrpId);
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		
		if (log.isDebugEnabled())
			log.debug("target filter [{}]", this.filterMap.toString());
		
		Set<String> channelCodeSet = this.filterMap.keySet();
		
		for (String channelCode : channelCodeSet) {
			log.info("update FilterCount [channelCode:{}, msgId:{}, postId:{}, filterCnt:{}]", channelCode, msgId, postId, this.filterMap.get(channelCode));
			jdbcTemplate.update(this.updateFilterCount,
					this.filterMap.get(channelCode),
					0,
					0,
					msgId);
		}
		
		return stepExecution.getExitStatus();
	}
}

