package com.humuson.tms.batch.item.database;

import static com.humuson.tms.batch.job.constrants.JobParamConstrants.APP_GRP_KEY;

import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemProcessor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;

import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.batch.service.PushResultService;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.PushResponseConstants;
import com.humuson.tms.constrants.StatusType;

import lombok.extern.slf4j.Slf4j;

/**
 * 각 채널 타겟팅 추출 데이터에 대한 검증 클래스
 * @author hyogun
 *
 */
@Slf4j
public class PushSendProcessor implements ItemProcessor<PushQueue, PushQueue>, StepExecutionListener {

	static final String HEAD = "<head>";
	static final String MAPPING_MSG_ID_KEYWORD 		= "[$]\\{MSG_ID\\}";
	static final String MAPPING_TYPE_KEYWORD 		= "[$]\\{TYPE\\}";
	
	@Autowired protected JdbcTemplate jdbcTemplate;
	@Autowired protected PushResultService pushResultService;
	@Autowired protected PushInfoService<App, PushMessage> pushInfoService;
	
	@Value("#{config['rich.content.charset.header']}")
	protected String charsetHeader;
	@Value("#{config['rich.content.msg.id.header']}")
	protected String msgIdHeader;
	@Value("#{config['rich.content.push.type.header']}")
	protected String pushTypeHeader;
	
	protected Map<String, Integer> filterMap = new ConcurrentHashMap<String, Integer>();
	protected Map<String, Integer> readerMap = new ConcurrentHashMap<String, Integer>();
	protected Map<String, Integer> writerMap = new ConcurrentHashMap<String, Integer>();
	protected Map<String, Integer> errorMap = new ConcurrentHashMap<String, Integer>();
	protected Map<Integer, String> appOsMap = new ConcurrentHashMap<Integer, String>();
	protected AtomicInteger readerCount = new AtomicInteger();
	protected AtomicInteger errorCount = new AtomicInteger();
	
	protected App appInfo;
	protected int maxChunkSize = 10;
	private Random rnd = new Random();
	
	@Value("#{config['use.mgs.public.push']}")
	protected boolean useMgsPublicPush;
	
	protected boolean checkValidation(PushQueue item) {
		boolean isValid = true;
		if (item.getAppId() == -1 || item.getDeviceId() == -1) {
			return false;
		}
		return isValid;
	}
	
	@Override
	public PushQueue process(PushQueue item) throws Exception {
		readerCount.incrementAndGet();
		
		item.setChunkId(String.format("%02d", rnd.nextInt(10)+1));
		String os = appOsMap.get(item.getAppId());
		if (StringUtils.isNull(os)) {
			os = App.UNKNOWN;
		}
		item.setAppOs(os);
		
		if (!this.checkValidation(item)) {
			item.setErrorCode(PushResponseConstants.NO_PMS_USER);
			pushResultService.updatePushQueue(item);
			return null;
		}
		
		if (useMgsPublicPush) {
			item.setSendStatus(StatusType.MQ_SENDING.getCode());
			item.setErrorCode(PushResponseConstants.MQ_SENDING);
		}
		
		
		if ("H".equals(item.getPushMessage().getMsgType())) {
			
			String parsedMsgContent = null;
			
			if (!StringUtils.isNull(item.getPushMessage().getPopupContent())) {
				parsedMsgContent = makeRichContents(item.getPushMessage().getPopupContent(), "P", item.getPushMessage().getMsgUid());
				item.getPushMessage().setPopupContent(parsedMsgContent);
			}
			
			if (!StringUtils.isNull(item.getPushMessage().getInAppContent())) {
				parsedMsgContent = makeRichContents(item.getPushMessage().getInAppContent(), "M", item.getPushMessage().getMsgUid());
				item.getPushMessage().setInAppContent(parsedMsgContent);
			}
			
			if (log.isDebugEnabled()) {
				log.debug("item.getPushMessage().getPopupContent() : {}", item.getPushMessage().getPopupContent());
				log.debug("item.getPushMessage().getInAppContent() : {}", item.getPushMessage().getInAppContent());
			}
		}
		
		return item;
	}
	
	/**
	 * 
	 * @param oriContent
	 * @param msgType P : PopupMsg, M : InAppMsg
	 * @param msgId
	 * @return
	 */
	public String makeRichContents(String oriContent, String msgType, String msgId) {
		StringBuilder richContentBuf = new StringBuilder();
		
		String parsedMsgIdHeader = msgIdHeader.replace(MAPPING_MSG_ID_KEYWORD, msgId);
		String richHeadBefore = oriContent.substring(0, oriContent.indexOf(HEAD)+ HEAD.length());
		richContentBuf.append(richHeadBefore).append("\n");
		richContentBuf.append(charsetHeader);
		richContentBuf.append("\n");
		richContentBuf.append(parsedMsgIdHeader);
		richContentBuf.append("\n");
		richContentBuf.append(pushTypeHeader.replace(MAPPING_TYPE_KEYWORD, msgType));
		String richHeadAfter = oriContent.substring(oriContent.indexOf(HEAD) + HEAD.length(),
				oriContent.length());
		richContentBuf.append(richHeadAfter);
		
		return richContentBuf.toString();
	}
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		String appGrpKey = stepExecution.getJobExecution().getJobParameters().getString(APP_GRP_KEY);
		appInfo = pushInfoService.getAppInfo(appGrpKey);
		appOsMap = appInfo.getAppOsMap();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		
		if (log.isDebugEnabled())
			log.debug("target filter [{}]", this.filterMap.toString());
		
		return stepExecution.getExitStatus();
	}
}

