package com.humuson.tms.batch.item.database;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.MqProducer;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.batch.service.PushResultService;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.PushChnType;
import com.humuson.tms.mq.model.MgsPush.Response.ResponsePayload;
import com.humuson.tms.mq.model.MgsPush.Response.ResultCode;

import lombok.extern.slf4j.Slf4j;

/**
 * rainboots의 푸시 전송 결과를 PMS DB에 반영하는 클래스 
 * 푸시 전송 상세 데이터 반영 : TB_SEND_RAW (RCV_TIME, PUSH_RESULT_CD)
 * 푸시 전송 통계 데이터 반영 : TB_SEND_SCHDL 
 * 푸시 전송 실패시 PUBLIC(GCM)으로의 발송은 성능상의 이슈로 사용하지 않고 구현만 해놓음
 * @author hyogun
 *
 */
@Slf4j
public class PushFeedbackItemWriter implements ItemWriter<PushResult> {
	
	protected static final String USE = "Y";

	@Autowired protected PushResultService pushResultService;
	
	@Autowired
	@Qualifier(value = "mqCampResProducer")
	private MqProducer mqCampResProducer;
	
	@Autowired
	@Qualifier(value = "mqAutoResProducer")
	private MqProducer mqAutoResProducer;
	
	@Autowired protected PushInfoService<App, PushMessage> pushInfoService;
	
	/**
	 * 
	 */
	@Override
	public void write(List<? extends PushResult> rawData) throws Exception {
		
		if (log.isDebugEnabled())
			log.debug("feedback write size : {}",rawData.size());
		@SuppressWarnings("unchecked")
		ArrayList<PushResult> prvPushResultList = new ArrayList<PushResult>();
		
		prvPushResultList.addAll(rawData);
		
		List<Object[]> prvPushMqDeleteList = new ArrayList<Object[]>();
		
		
		String appKey = pushInfoService.selectAndroidAppKeyByAppGrpId(prvPushResultList.get(0).getAppGrpId()); 
		
		
		MgsPush.Response.Builder mqAutoResponseBuilder = MgsPush.Response.newBuilder();
		MgsPush.Response.Builder mqCampResponseBuilder = MgsPush.Response.newBuilder();
		ResponsePayload.Builder  responsePayloadBuilder = ResponsePayload.newBuilder();
		
		for(Iterator<PushResult> it = prvPushResultList.iterator(); it.hasNext();){
			PushResult result = it.next(); 
			if( result.isMqSending()){
				it.remove();
				responsePayloadBuilder.clear();
				//pushId+'_'+deviceId+'_'+reqUid+'_'+custId
				responsePayloadBuilder.setId(result.getPushId()+"&&"+result.getDeviceId()+"&&"+result.getReqUid()+"&&"+result.getCustId());
				responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.SUCCESSFUL);
				responsePayloadBuilder.setServerId(result.getServerId());
				if(result.getReqUid().startsWith("AA")){
					mqAutoResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}else if(result.getReqUid().startsWith("CC")){
					mqCampResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}
				prvPushMqDeleteList.add(new Object[]{result.getReqUid()});
			}
		}
		
		if(!mqAutoResponseBuilder.getResPayloadList().isEmpty() ){
			mqAutoResponseBuilder.setResultCode(ResultCode.SUCCESS);
			mqAutoResponseBuilder.setAppKey(appKey);
			mqAutoResponseBuilder.setPushChnType(PushChnType.GCM);
			mqAutoResProducer.send(mqAutoResponseBuilder.build());
		}
		
		if(!mqCampResponseBuilder.getResPayloadList().isEmpty() ){
			mqCampResponseBuilder.setResultCode(ResultCode.SUCCESS);
			mqCampResponseBuilder.setAppKey(appKey);
			mqCampResponseBuilder.setPushChnType(PushChnType.GCM);
			mqCampResProducer.send(mqCampResponseBuilder.build());
		}
		
		if(!prvPushResultList.isEmpty()){
			pushResultService.batchUpdatePushResult(prvPushResultList);
		}
		
		if(!prvPushMqDeleteList.isEmpty()){
			pushResultService.deleteMqResendList(prvPushMqDeleteList);
		}
	}
}
