package com.humuson.tms.batch.item.database;

import static com.humuson.tms.batch.job.constrants.JobParamConstrants.MSG_ID;

import java.util.Random;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemProcessor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.batch.service.SystemCodeService;
import com.humuson.tms.common.model.target.TargetData;
import com.humuson.tms.common.security.HumusonDecryptor;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.ChannelType;

import lombok.extern.slf4j.Slf4j;

/**
 * 각 채널 타겟팅 추출 데이터에 대한 검증 클래스
 * @author hyogun
 *
 */
@Slf4j
public class OptmzTargetProcessor implements ItemProcessor<TargetData, TargetData>, StepExecutionListener {
	
	protected String chnType, postId, targetType, idTargetYn, deduplicationYn, msgId;
	protected Long siteId;
	
	private int maxChunkSize = 10;
	private Random rnd = new Random();
	@Value("#{config['tms.db.enc.key']}") protected String encKey;
	protected String dbEncryptFlag;
	@Autowired SystemCodeService systemCodeServiceImpl;
	
	@Override
	public TargetData process(TargetData item) throws Exception {
		
		if (ChannelType.EMAIL.getCode().equals(chnType)) {
			String email = item.getMemberEmail();
			if (StringUtils.isNull(email)) {
				return null;
			} else {
				if ("Y".equals(dbEncryptFlag)) {
					try {
						email = HumusonDecryptor.decrypt(email, encKey, false);
					} catch (Exception e) {
						log.error("email decrypt fail {}", email);
					}
				}
				if (StringUtils.isEmailError(email)) {
					return null;
				}
				
				item.setDomain(StringUtils.getDomain(email));
			}
		} else if (ChannelType.PUSH.getCode().equals(chnType)) {
			if (StringUtils.isNull(item.getMemberToken())
					|| StringUtils.isNull(item.getDeviceId())) {
				if (log.isDebugEnabled()) {
					log.debug("pushToken or deviceId is null [{}]", item.toString());
				}
				return null;
			}
		} else if (ChannelType.SMS.getCode().equals(chnType)) {
			String phone = item.getMemberPhone();
			if (StringUtils.isNull(phone)) {
				return null;
			} else {
				if ("Y".equals(dbEncryptFlag)) {
					try {
						phone = HumusonDecryptor.decrypt(phone, encKey, false);
					} catch (Exception e) {
						log.error("phone decrypt fail {}", phone);
					}
				}
				
				if (StringUtils.isSmsError(phone)) {
					return null;
				} else {
					item.setTelCode(StringUtils.getTeleCom(phone));
				}
			}
		}
		
		item.setChannelType(chnType)
		.setPostId(postId)
		.setChunkId(String.valueOf(rnd.nextInt(maxChunkSize)))
		.setMsgId(msgId);
		
		if ("Y".equals(deduplicationYn)) {
			if (ChannelType.EMAIL.getCode().equals(chnType)) {
				item.setDuplicateKey(item.getMemberEmail());
			} else if (ChannelType.SMS.getCode().equals(chnType)) {
				item.setDuplicateKey(item.getMemberPhone());
			} else if (ChannelType.PUSH.getCode().equals(chnType)) {
				item.setDuplicateKey(item.getDeviceId());
			}
		}
		
		return item;
	}
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		chnType = stepExecution.getJobParameters().getString(JobParamConstrants.CHN_TYPE);
		msgId = stepExecution.getJobParameters().getString(MSG_ID);
		postId = stepExecution.getJobParameters().getString(JobParamConstrants.POST_ID);
		siteId = stepExecution.getJobParameters().getLong(JobParamConstrants.SITE_ID);		
		log.info("chnType:{}, siteId:{}, postId:{}", chnType, siteId, postId);
		dbEncryptFlag = systemCodeServiceImpl.getDbEncryptYn();
		deduplicationYn = stepExecution.getJobParameters().getString(JobParamConstrants.DEDUPLICATION_YN, "N");
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		
		return stepExecution.getExitStatus();
	}

}
