package com.humuson.tms.batch.item.apns;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import javapns.Push;
import javapns.devices.Device;
import lombok.extern.slf4j.Slf4j;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.StringUtils;

import com.humuson.tms.batch.domain.ApnsFeedback;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.common.security.HumusonDecryptor;
import com.humuson.tms.common.util.FileUtil;

/**
 * Apns Feedback 정보를 가져옴
 * @author hyogun
 *
 */
@Slf4j
public class ApnsFeedbackItemReader implements ItemReader<ApnsFeedback>,  StepExecutionListener {
	
	private List<ApnsFeedback> results;
	
	private int currentReadCount = 0;
	
	private int currentItemCount = 0;
	
	private int maxItemCount = Integer.MAX_VALUE;
	
	@Value("#{config['tms.db.enc.key']}") protected String encKey;
	protected App appInfo;
	
	@Autowired
	private PushInfoService<App, PushMessage> pushInfoServiceImpl;
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		String appGrpKey = stepExecution.getJobParameters().getString(JobParamConstrants.APP_GRP_KEY);
		appInfo = pushInfoServiceImpl.getAppInfo(appGrpKey);
		log.info("appInfo : {}", appInfo.toString());
	}
	
	@Override
	public ApnsFeedback read() throws Exception {
		if (currentItemCount >= maxItemCount) {
			return null;
		}
		
		return doRead();
	}
	
	private synchronized ApnsFeedback doRead() throws Exception {
		if (results == null || currentReadCount >= currentItemCount) {
			
			doMoreFeedbackList();
			
			if (results != null
					&& log.isDebugEnabled()) {
				log.debug("reader size : {}", currentItemCount);
			}
		}
		
		int next = currentReadCount++;
		
		if (results != null && next < results.size()) {
			return results.get(next);
		} else {
			return null;
		}
	}
	
	private void doMoreFeedbackList() {
		
		if (results == null) {
			results = new CopyOnWriteArrayList<ApnsFeedback>();
			currentItemCount = 0;
		} 
			
		String apnsCert = appInfo.getApnsPushCert();
		String apnsPwd = appInfo.getApnsPushPwd();
		
		if (!StringUtils.isEmpty(apnsCert) && !StringUtils.isEmpty(apnsPwd)) {
			
			if (FileUtil.isValidFile(apnsCert)) {
				
				try {
					apnsPwd = HumusonDecryptor.decrypt(apnsPwd, encKey);
					
				} catch (Exception e) {
					log.error("HumusonDecryptor exception : {}", e);
				}
				
				try {
					List<Device> feedbackList = Push.feedback(apnsCert, apnsPwd, true);
					
					for (Device device : feedbackList) {
						log.debug("APNS Feedback [appGrpId:{}, token:{}, deviceId:{}]", appInfo.getAppGrpId(),
								device.getToken(), device.getDeviceId());
						results.add(new ApnsFeedback(appInfo.getAppGrpId(), device.getToken()));
						currentItemCount++;
					}
				} catch (Exception e) {
					log.error("Apns error[apnsCert:{}, apnsPwd:{}] : {}", apnsCert, apnsPwd, e);
				}
			} else {
				log.error("apns p12 file validation error [file:{}, pwd:{}]", apnsCert, apnsPwd);
			}
		} else {
			log.error("apns file or pwd is null");
			throw new NullPointerException("apns file or pwd is null");
		}
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		log.info("APNS Feedback read count : {}, currentReadCount : {}, currentItemCount : {} result size : {}", 
				stepExecution.getReadCount(), currentReadCount, currentItemCount, results == null ? 0 : results.size());
		return stepExecution.getExitStatus();
	}
}