package com.google.android.gcm.ccs.server;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;

import org.jivesoftware.smack.SmackException.NotConnectedException;
import org.jivesoftware.smack.StanzaListener;
import org.jivesoftware.smack.packet.Message;
import org.jivesoftware.smack.packet.Stanza;
import org.json.simple.JSONValue;
import org.json.simple.parser.ParseException;

import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.job.constrants.GCMConstants;
import com.humuson.tms.batch.service.MqProducer;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.batch.service.PushResultService;
import com.humuson.tms.common.model.CcsMessageId;
import com.humuson.tms.common.util.PushCcsMessageIdUtil;
import com.humuson.tms.constrants.CommonType;
import com.humuson.tms.constrants.PushResponseConstants;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.PushChnType;
import com.humuson.tms.mq.model.MgsPush.Response.ResponsePayload;
import com.humuson.tms.mq.model.MgsPush.Response.ResultCode;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class CcsStanzaListener implements StanzaListener {

	XMPPGCMConnection connection;
	
	private MqProducer mqCampResProducer;

	private MqProducer mqAutoResProducer;
	
	PushInfoService<App, PushMessage> pushInfoServiceImpl;

	PushResultService pushResultSeviceImpl;
	
	private boolean upstreamUseFlag;
	
	private static final String MESSAGE_TYPE_UPSTREAM = "upstream";
	private static final String MESSAGE_TYPE_RECEIPT = "receipt";
	
	public CcsStanzaListener(XMPPGCMConnection connection, 
			MqProducer mqCampResProducer, MqProducer mqAutoResProducer,
			PushInfoService<App, PushMessage> pushInfoServiceImpl,
			PushResultService pushResultSeviceImpl,
			boolean upstreamUseFlag) {
		this.connection = connection;
		this.mqCampResProducer = mqCampResProducer;
		this.mqAutoResProducer = mqAutoResProducer;
		this.pushInfoServiceImpl = pushInfoServiceImpl;
		this.pushResultSeviceImpl = pushResultSeviceImpl;
		this.upstreamUseFlag = upstreamUseFlag;
	}
	
	
	@Override
	public void processPacket(Stanza packet) throws NotConnectedException {
		//&quot; -> "
		//&amp; -> &
//		log.info("[CcsStanzaListener] Received: " + (packet.toXML()).toString().replaceAll("&quot;", "\"").replaceAll("&amp;", "&"));
		log.debug("[CcsStanzaListener] Received: " + packet.toXML());
        Message incomingMessage = (Message) packet;
        GcmPacketExtension gcmPacket =
                (GcmPacketExtension) incomingMessage.
                getExtension(GCMConstants.GCM_NAMESPACE);
        String json = gcmPacket.getJson();
        
        try {
            @SuppressWarnings("unchecked")
            Map<String, Object> jsonObject =
                    (Map<String, Object>) JSONValue.
                    parseWithException(json);

            // present for "ack"/"nack", null otherwise
            Object messageType = jsonObject.get("message_type");

            if (messageType == null) {
//            	log.info("[CcsStanzaListener] upstream process");
                // Normal upstream data message
//                handleUpstreamMessage(jsonObject);
            	if(upstreamUseFlag) handleUpstream(jsonObject, MESSAGE_TYPE_UPSTREAM);

                // Send ACK to CCS
                String messageId = (String) jsonObject.get("message_id");
                String from = (String) jsonObject.get("from");
                String ack = createJsonAck(from, messageId);
                send(ack);
            } else if ("ack".equals(messageType.toString())) {
                  // Process Ack
                  handleAckReceipt(jsonObject);
            } else if ("nack".equals(messageType.toString())) {
                  // Process Nack
                  handleNackReceipt(jsonObject);
            } else if ("control".equals(messageType.toString())) {
                  // Process control message
                  handleControlMessage(jsonObject);
            } else if ("receipt".equals(messageType.toString())) {
            	// 임시 테스트 용도. 운영에서는 제거
//            	log.info("[CcsStanzaListener] receipt process");
//            	if(upstreamUseFlag) handleUpstream(jsonObject, MESSAGE_TYPE_RECEIPT);
            } else {
                  log.warn("Unrecognized message type {}",messageType.toString());
            }
        } catch (ParseException e) {
            log.error( "Error parsing JSON {} / {}" , json, e);
        } catch (Exception e) {
        	 log.error( "Failed to process packet {}" , e);
        }
		
	}
	
	/**
     * Sends a downstream message to GCM.
     *
     * @return true if the message has been successfully sent.
     */
    public boolean sendDownstreamMessage(String jsonRequest) throws
            NotConnectedException {
        if (!connection.isDrainning()) {
            send(jsonRequest);
            return true;
        }
        log.debug("Dropping downstream message since the connection is draining");
        return false;
    }

    /**
     * Returns a random message id to uniquely identify a message.
     *
     * <p>Note: This is generated by a pseudo random number generator for
     * illustration purpose, and is not guaranteed to be unique.
     */
    public String nextMessageId() {
        return "m-" + UUID.randomUUID().toString();
    }

    /**
     * Sends a packet with contents provided.
     */
    protected void send(String jsonRequest) throws NotConnectedException {
        Stanza request = new GcmPacketExtension(jsonRequest).toPacket();
        connection.sendStanza(request);
    }

    /**
     * Handles an upstream data message from a device application.
     *
     * <p>This sample echo server sends an echo message back to the device.
     * Subclasses should override this method to properly process upstream messages.
     */
    protected void handleUpstreamMessage(Map<String, Object> jsonObject) {
        // PackageName of the application that sent this message.
        String category = (String) jsonObject.get("category");
        String from = (String) jsonObject.get("from");
        log.debug("[handleUpstreamMessage] category: {} / from: {}",category,from);
        
        @SuppressWarnings("unchecked")
        Map<String, String> payload = (Map<String, String>) jsonObject.get("data");
        
        Iterator<String> keys = payload.keySet().iterator();
        while( keys.hasNext() ){
            String key = keys.next();
            log.debug( String.format("키 : %s, 값 : %s", key, payload.get(key)) );
        }
        
        /*
        payload.put("ECHO", "Application: " + category);
        // Send an ECHO response back
        String echo = createJsonMessage(from, nextMessageId(), payload,
                "echo:CollapseKey", null, false);
        log.info(echo);
        try {
            sendDownstreamMessage(echo);
        } catch (NotConnectedException e) {
            log.warn("Not connected anymore, echo message is not sent", e);
        }
        */
    }

    /**
     * Handles an ACK.
     *
     * <p>Logs a INFO message, but subclasses could override it to
     * properly handle ACKs.
     */
    protected void handleAckReceipt(Map<String, Object> jsonObject) {
    	MgsPush.Response.Builder responseBuilder        = null;
    	ResponsePayload.Builder  responsePayloadBuilder = null;
    	
    	String messageId = (String) jsonObject.get("message_id");
       String from = (String) jsonObject.get("from");
       String splitMsgId[] = messageId.split(":");
       
       CcsMessageId msg = PushCcsMessageIdUtil.parseCcsMessageId(messageId);
       
       //pushInfo : pushId+"&&"+deviceId +"&&"+pushQueue.getReqUid()+"&&"+pushQueue.getCustId()
        
       //테스트 발송 결과 처리 안함.
       if(CommonType.TEST.equals(msg.getCommonSendType()) && msg.isMqSending()){
   			return;
   		}
       
        log.debug("recieveMsg {} ", jsonObject);
        try {
        	if( msg.isMqSending() ){
        		responseBuilder        = MgsPush.Response.newBuilder();
            	responsePayloadBuilder = ResponsePayload.newBuilder();
	        	responsePayloadBuilder.clear();
	        	responseBuilder.clear();
	        	responseBuilder.setAppKey(pushInfoServiceImpl.selectAndroidAppKeyByAppGrpId(msg.getAppGrpId()))
				   .setPushChnType(PushChnType.GCM)
				   .setResultCode(ResultCode.SUCCESS);
	    		responsePayloadBuilder.setId(msg.getId()) 
	    							  .setReturnCode(MgsPush.Response.ReturnCode.SUCCESSFUL)
	    							  .setServerId(msg.getServerId());
	    		responseBuilder.addResPayload(responsePayloadBuilder.build());
	    		if(CommonType.AUTO.equals(msg.getCommonSendType())){
	    			mqAutoResProducer.send(responseBuilder.build());
	    			log.debug("handle Ack Auto Response que insert que name : {}", mqAutoResProducer.getDefaultDestination());
	    		}else if (CommonType.CAMP.equals(msg.getCommonSendType())){
	    			mqCampResProducer.send(responseBuilder.build());
	    			log.debug("handle Ack Camp Response que insert que name : {}", mqCampResProducer.getDefaultDestination());
	    		}
	    		pushResultSeviceImpl.deleteMqResend(msg.getReqUid());
        	}else{
        		pushResultSeviceImpl.deleteInsertPushResult(msg,PushResponseConstants.SUCCESSFUL);
        		log.debug("handle Ack insert que_log");
        	}
		} catch (Exception e) {
			log.error("handleAckReceipt() error ", e);
			if(msg.isMqSending()){
				log.error("Re Attempt Response send to MQ");
				if(CommonType.AUTO.equals(msg.getCommonSendType())){
	    			mqAutoResProducer.send(responseBuilder.build());
	    		}else if (CommonType.CAMP.equals(msg.getCommonSendType())){
	    			mqCampResProducer.send(responseBuilder.build());
	    		}
			}else{
				log.error("GCM CCS Response processing Error(deleteInsertPushResult) / pushId :{} / error : {}",msg.getPushId(), e );
			}
		}finally{
			responseBuilder = null;
			responsePayloadBuilder = null;
		}
    }

    /**
     * Handles a NACK.
     *
     * <p>Logs a INFO message, but subclasses could override it to
     * properly handle NACKs.
     */
    protected void handleNackReceipt(Map<String, Object> jsonObject) {
    	MgsPush.Response.Builder responseBuilder        = null;
    	ResponsePayload.Builder  responsePayloadBuilder = null;
    	
    	String messageId = (String) jsonObject.get("message_id");
    	CcsMessageId msg = PushCcsMessageIdUtil.parseCcsMessageId(messageId);
        
       //pushInfo : pushId+"&&"+deviceId +"&&"+pushQueue.getReqUid()+"&&"+pushQueue.getCustId()
         
        if(CommonType.TEST.equals(msg.getCommonSendType())&& msg.isMqSending()){
			return;
		}
        
        log.debug("recieveMsg {} ", jsonObject);
        String from = (String) jsonObject.get("from");
        String error = (String) jsonObject.get("error");	
         
        
        String resultCd = null;
        if (error.equals("BAD_ACK")){
        	resultCd = PushResponseConstants.BAD_ACK;        
        } else if (error.equals("BAD_REGISTRATION")){
        	resultCd = PushResponseConstants.ERROR_INVALID_REGISTRATION;        	
        } else if (error.equals("CONNECTION_DRAINING")){
        	resultCd = PushResponseConstants.ERROR_UNAVAILABLE;        	
        } else if (error.equals("DEVICE_UNREGISTERED")){
        	resultCd = PushResponseConstants.ERROR_NOT_REGISTERED;        	
        } else if (error.equals("INTERNAL_SERVER_ERROR")){
        	resultCd = PushResponseConstants.ERROR_UNAVAILABLE;        	
        } else if (error.equals("INVALID_JSON")){
        	resultCd = PushResponseConstants.INVALID_JSON;
        } else if (error.equals("DEVICE_MESSAGE_RATE_EXCEEDED")){
        	resultCd = PushResponseConstants.ERROR_DEVICE_QUOTA_EXCEEDED;        	
        } else if (error.equals("SERVICE_UNAVAILABLE")){
        	resultCd = PushResponseConstants.ERROR_QUOTA_EXCEEDED;        	
        } else{
        	resultCd = PushResponseConstants.ERROR_UNAVAILABLE;
        }
        
        try {
        	if(msg.isMqSending()){
        		responseBuilder        = MgsPush.Response.newBuilder();
            	responsePayloadBuilder = ResponsePayload.newBuilder();
	        	responsePayloadBuilder.clear();
	        	responseBuilder.clear();
	        	responseBuilder.setAppKey(pushInfoServiceImpl.selectAndroidAppKeyByAppGrpId(msg.getAppGrpId()))
				   .setPushChnType(PushChnType.GCM)
				   .setResultCode(ResultCode.SUCCESS);
	    		responsePayloadBuilder.setId(msg.getId()) 
	    							  .setReturnCode(MgsPush.Response.ReturnCode.valueOf(Integer.parseInt(resultCd)))
	    							  .setServerId(msg.getServerId());
	    		responseBuilder.addResPayload(responsePayloadBuilder.build());
	    		
	    		if(CommonType.AUTO.equals(msg.getCommonSendType())){
	    			mqAutoResProducer.send(responseBuilder.build());
	    			log.debug("handle NACK {} Auto Response que insert que name : {}",resultCd, mqCampResProducer.getDefaultDestination());
	    		}else if (CommonType.CAMP.equals(msg.getCommonSendType())){
	    			mqCampResProducer.send(responseBuilder.build());
	    			log.debug("handle NACK {} Camp Response que insert que name : {}",resultCd, mqCampResProducer.getDefaultDestination());
	    		}
	    		pushResultSeviceImpl.deleteMqResend(msg.getReqUid());
        	}else{
        		pushResultSeviceImpl.deleteInsertPushResult(msg,resultCd);
        		log.debug("handle NACK Response deleteInsertPushResult");
        	}
		} catch (Exception e) {
			log.error("handleNackReceipt()-{} error : {}", messageId, e);
			if(msg.isMqSending()){
				log.error("Re Attempt Response send to MQ");
				if(CommonType.AUTO.equals(msg.getCommonSendType())){
	    			mqAutoResProducer.send(responseBuilder.build());
	    		}else if (CommonType.CAMP.equals(msg.getCommonSendType())){
	    			mqCampResProducer.send(responseBuilder.build());
	    		}
			}else{
				log.error("GCM CCS Response processing Error(deleteInsertPushResult) / pushId :{} / error : {}",msg.getPushId(), e );
			}
		}finally{
			responseBuilder = null;
			responsePayloadBuilder = null;
		}
    }

    protected void handleControlMessage(Map<String, Object> jsonObject) {
    	log.debug("handleControlMessage(): {}", jsonObject);
        String controlType = (String) jsonObject.get("control_type");
 
//        con.setDrainning(true);
//        con.disconnect();
        if ("CONNECTION_DRAINING".equals(controlType)) {
//            connectionDraining = true;
//        	gcmCcsService.setConnectionDraining(true);
        	connection.setDrainning(true);
        	log.info("CONNECTION_DRAINING!!!!!!!!!!!!!");
        } else {
            log.info("Unrecognized control type: {}. This could happen if new features are added to the CCS protocol.", controlType);
        }
    }

    /**
     * Creates a JSON encoded GCM message.
     *
     * @param to RegistrationId of the target device (Required).
     * @param messageId Unique messageId for which CCS sends an
     *         "ack/nack" (Required).
     * @param payload Message content intended for the application. (Optional).
     * @param collapseKey GCM collapse_key parameter (Optional).
     * @param timeToLive GCM time_to_live parameter (Optional).
     * @param delayWhileIdle GCM delay_while_idle parameter (Optional).
     * @return JSON encoded GCM message.
     */
    public static String createJsonMessage(String to, String messageId,
            Map<String, String> payload, String collapseKey, Long timeToLive,
            Boolean delayWhileIdle) {
        Map<String, Object> message = new HashMap<String, Object>();
        message.put("to", to);
        if (collapseKey != null) {
            message.put("collapse_key", collapseKey);
        }
        if (timeToLive != null) {
            message.put("time_to_live", timeToLive);
        }
        if (delayWhileIdle != null && delayWhileIdle) {
            message.put("delay_while_idle", true);
        }
      message.put("message_id", messageId);
      message.put("data", payload);
      return JSONValue.toJSONString(message);
    }

    /**
     * Creates a JSON encoded ACK message for an upstream message received
     * from an application.
     *
     * @param to RegistrationId of the device who sent the upstream message.
     * @param messageId messageId of the upstream message to be acknowledged to CCS.
     * @return JSON encoded ack.
     */
    protected static String createJsonAck(String to, String messageId) {
        Map<String, Object> message = new HashMap<String, Object>();
        message.put("message_type", "ack");
        message.put("to", to);
        message.put("message_id", messageId);
        return JSONValue.toJSONString(message);
    }
    
    protected void handleUpstream(Map<String, Object> jsonObject, String mType) {
    	MgsPush.Response.Builder responseBuilder        = null;
    	ResponsePayload.Builder  responsePayloadBuilder = null;
    	
    	String messageId = (String) jsonObject.get("message_id");
    	log.debug("handleUpstream: {}", messageId);
//    	CcsMessageId msg = parseUpstreamMessageId(messageId, mType);
    	//pushInfo : pushId+"&&"+deviceId +"&&"+pushQueue.getReqUid()+"&&"+pushQueue.getCustId()
    	CcsMessageId msg = null;
    	try {
			msg = parseUpstreamMessageId(messageId, mType);
		} catch (Exception e1) {
//			log.error("handle {} parsing ERROR {}", mType, e1);
		}
    	
    	if (msg != null){
    		
    		//resDate
    		@SuppressWarnings("unchecked")
    		Map<String, String> payload = (Map<String, String>) jsonObject.get("data");
    		String resDate = payload.get("res_date");
    		if (resDate == null) resDate = new java.text.SimpleDateFormat("yyyyMMddHHmmss").format(new java.util.Date());
    		
    		//테스트 발송 결과 처리 안함.
    		if(CommonType.TEST.equals(msg.getCommonSendType()) && msg.isMqSending()){
    			log.debug("this upsteam is TEST {}", messageId);
    			return;
    		}
    		
    		try {
//    			log.info("type "+msg.isMqSending());
    			if( msg.isMqSending() ){
    				//MQ 사용 시 도달 정보를 MQ pop 
    				responseBuilder        = MgsPush.Response.newBuilder();
    				responsePayloadBuilder = ResponsePayload.newBuilder();
    				responsePayloadBuilder.clear();
    				responseBuilder.clear();
    				responseBuilder.setAppKey(pushInfoServiceImpl.selectAndroidAppKeyByAppGrpId(msg.getAppGrpId()))
    				.setPushChnType(PushChnType.GCM)
    				.setResultCode(ResultCode.SUCCESS);
    				responsePayloadBuilder.setId(msg.getId()) 
    				.setReturnCode(MgsPush.Response.ReturnCode.XMPP_DELIVER)
    				.setServerId(msg.getServerId());
    				responseBuilder.addResPayload(responsePayloadBuilder.build());
    				if(CommonType.AUTO.equals(msg.getCommonSendType())){
    					mqAutoResProducer.send(responseBuilder.build());
    					log.debug("handle {} Auto Response que insert que name : {}", mType, mqAutoResProducer.getDefaultDestination());
    				}else if (CommonType.CAMP.equals(msg.getCommonSendType())){
    					mqCampResProducer.send(responseBuilder.build());
    					log.debug("handle {} Camp Response que insert que name : {}", mType, mqCampResProducer.getDefaultDestination());
    				}
    			}else{
    				//MQ 비사용 시 직접 send_list table update 
    				pushResultSeviceImpl.insertPushDeliver(msg,PushResponseConstants.XMPP_DELIVER, resDate);
    				log.debug("handle {} insert push_que_log", mType);
    			}
    		} catch (Exception e) {
    			log.error("handleAckReceipt() error ", e);
    			if(msg.isMqSending()){
    				log.error("Re Attempt Response send to MQ");
    				if(CommonType.AUTO.equals(msg.getCommonSendType())){
    					mqAutoResProducer.send(responseBuilder.build());
    				}else if (CommonType.CAMP.equals(msg.getCommonSendType())){
    					mqCampResProducer.send(responseBuilder.build());
    				}
    			}else{
    				log.error("GCM CCS Response processing Error(deleteInsertPushResult) / pushId :{} / error : {}",msg.getPushId(), e );
    			}
    		}finally{
    			responseBuilder = null;
    			responsePayloadBuilder = null;
    		}
    	}
        
    }
    
    public static CcsMessageId parseUpstreamMessageId(String messageId, String messageType){
    	String splitOriMsgId[] = null;
    	
    	if (MESSAGE_TYPE_RECEIPT.equals(messageType)) {
    		splitOriMsgId = messageId.split("dr2:");
        } else if (MESSAGE_TYPE_UPSTREAM.equals(messageType)) {
        	splitOriMsgId = messageId.split("dr:");
        }
    	
//    	splitOriMsgId = messageId.split("dr:");
    	if(splitOriMsgId != null) messageId = splitOriMsgId[splitOriMsgId.length-1];
    	log.debug("{} messageId : {}", messageType, messageId);
		return PushCcsMessageIdUtil.parseCcsMessageId(messageId);
	}

}
