package com.apple.ios.apns;

import java.io.File;
import java.io.FileInputStream;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import com.humuson.tms.batch.domain.App;
import com.humuson.tms.common.security.HumusonDecryptor;
import com.humuson.tms.common.security.SSLChecker;
import com.humuson.tms.common.util.FileUtil;

import javapns.notification.AppleNotificationServer;
import javapns.notification.AppleNotificationServerBasicImpl;
import javapns.notification.PushNotificationManager;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class ApnsConnectionManager {

	@Setter
	private String selectIosAppInfo;
	
	@Value("#{config['tms.db.enc.key']}")
	protected String encKey;
	
	@Value("#{config['tms.apns.production.cn']}")
	protected String productionCN; // "Apple Push Services|Apple Production IOS Push Services"
	
	@Value("#{config['tms.apns.development.cn']}")
	protected String developmentCN; // "Apple Development IOS Push Services"
	
	@Value("#{config['ios.cert.path']}")
	protected String iosCertPath;
	
	 @Autowired
    JdbcTemplate jdbcTemplate;
	
	ConcurrentHashMap<String, PushNotificationManager> pushManagerConcurrentMap = new ConcurrentHashMap<String,PushNotificationManager>();
	
	public void init(){
		List <App> appInfoList = jdbcTemplate.query(selectIosAppInfo, new RowMapper<App>() {
			
			@Override
			public App mapRow(ResultSet rs, int rowNum) throws SQLException {
				// TODO Auto-generated method stub
				App app = new App();
				app.putAppKey(App.IOS,rs.getString(App.APP_KEY));
				app.setApnsPushCert(rs.getString(App.PUSH_CERT));
				app.setApnsPushPwd(rs.getString(App.PUSH_PWD));
				return app;
			}
		});
		
		for(App app : appInfoList){
			String apnsCert = iosCertPath.concat(File.separator).concat(app.getApnsPushCert().toString());
			String apnsPwd = app.getApnsPushPwd();
			String appKey = app.getAppKey(App.IOS);
			if (apnsCert != null && apnsPwd != null) {
				if (FileUtil.isValidFile(apnsCert)) {
					try {
						String decryptApnsPwd = HumusonDecryptor.decrypt(apnsPwd, encKey);
						PushNotificationManager pushManager =
								this.openApnsConnection(apnsCert, decryptApnsPwd, SSLChecker.isProduction(new FileInputStream(apnsCert), decryptApnsPwd, productionCN, developmentCN, false));
						if(pushManager != null){
							pushManagerConcurrentMap.put(appKey, pushManager);}
						else{
							log.error("apns manager put error : pushManager is null [appInfo:{}, encKey:{}]", app.toString(), encKey);
						}
					} catch (Exception e) {
						log.error("apns init exception [appInfo:{}, encKey:{}]", app.toString(), encKey, e);
					}
				} else {
					log.error("APNS Cert File {} is not valid ", apnsCert);
				}
			} else {
				log.error("APNS Cert is Null or ApnsPwd is Null {}", app.toString());
			}
		}
	}
	
	protected PushNotificationManager openApnsConnection (String apnsCertPath, String password, boolean isReal) {
		
		PushNotificationManager pushManager = null;
		try {
			AppleNotificationServer server = new AppleNotificationServerBasicImpl(apnsCertPath, password, isReal);
			pushManager = new PushNotificationManager();
			
			//connection open
			pushManager.initializeConnection(server);
			
		} catch (Exception e) {
			log.error("error : {}", e);
			try {
				if (pushManager != null) {
					pushManager.stopConnection();
				}
			} catch (Exception e2) {
				log.error("stop connection error", e2);
			} finally {
				pushManager = null;
			}
		}
		
		return pushManager;
	}
	
	public PushNotificationManager getPushNotificationManager(String appKey, String apnsCertPath, String password)
			throws Exception{
		PushNotificationManager pushManager = pushManagerConcurrentMap.get(appKey);
		
		if(pushManager == null){
			pushManager = openApnsConnection ( apnsCertPath,  password,  SSLChecker.isProduction(new FileInputStream(apnsCertPath), password, productionCN, developmentCN, false));
			if(pushManager != null){
				pushManagerConcurrentMap.put(appKey, pushManager);
			}else{
				throw new Exception("Push Manager Create Fail");
			}
		}
		
		return pushManager;
	}
}
